<?php

namespace CLF\Inc\Frontend;

use CLF\Inc\Common\Gateways;
use CLF\Inc\Common\CleverFinance_Blocks;

use WP_REST_Request;
use WP_REST_Response;
use Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry;

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *   
 */
class Frontend {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * The text domain of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_text_domain    The text domain of this plugin.
	 */
	private $plugin_text_domain;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since       1.0.0
	 * @param       string $plugin_name        The name of this plugin.
	 * @param       string $version            The version of this plugin.
	 * @param       string $plugin_text_domain The text domain of this plugin.
	 */
	public function __construct( $plugin_name, $version, $plugin_text_domain ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->plugin_text_domain = $plugin_text_domain;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/clf-frontend.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/clf-frontend.js', array( 'jquery' ), $this->version, false );
			
	}

	public function get_gateways( array $gateways ) {
		return array_merge( $gateways, Gateways::getGateways() );
	}
	public function enqueue_block_script() {
			wp_enqueue_script('clf_blocks_gateway_script', plugin_dir_url( __FILE__ ) . 'js/clf-blocks.js', ['wp-element', 'wc-blocks'], null, true);
			
			// Pass gateway settings to JS
			$gateways = Gateways::getGateways();
			$gateway_data = [];
			
			foreach ($gateways as $gateway_key => $gateway_class) {
				$gateway_instance = new $gateway_class();
				$settings = $gateway_instance->getSettings(); 
				$gateway_data[$gateway_key] = $settings;
			}
		
			wp_localize_script('clf_blocks_gateway_script', 'clf_blocks_gateway_data', $gateway_data);
	}
	public function register_clever_finance_payment_methods() {
        // Check if WooCommerce Blocks' AbstractPaymentMethodType class exists
        if ( ! class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
            return;
        }
        // Hook into WooCommerce Blocks payment method registration
		add_action( 
			'woocommerce_blocks_payment_method_type_registration',
			function ( PaymentMethodRegistry $payment_method_registry ) {
				// Get all available gateways
				$gateways = Gateways::getGateways();
				
				// Loop through each gateway and register a corresponding Blocks class
				foreach ( $gateways as $gateway_key => $gateway_class ) {
					// Check if the class exists before registering
					if ( class_exists( $gateway_class ) ) {
						// Create an instance of the payment method class (or its implementation)
						$gateway_instance = new $gateway_class(); 
						// Now pass the gateway instance to the block class
						$blocks_instance = new CleverFinance_Blocks($gateway_instance);
						// Register the blocks class with the payment method registry
						$payment_method_registry->register( $blocks_instance );
					}
				}
			}
		);
		
    }
    
	public function register_rest_route() {
		register_rest_route( 'digiteal/v1', '/payment', array(
			'methods' => 'POST',
			'callback' => array( $this, 'process_notification' ),
			'permission_callback' => '__return_true'
		) );
	}

	public function process_notification( WP_REST_Request $request ) {
		// Get the parameters
		$parameters = $request->get_params();

		// Get the payment status
		$status = $parameters['paymentStatus'] ?? null;

		// Get the payment internal ID
		$internal_id = $parameters['paymentRequestInformation']['paymentInternalId'] ?? null;

		// Get the payment ID
		$payment_id = $parameters['paymentRequestInformation']['paymentID'] ?? null;

		// Check if any of the parameters is null
		if ( is_null( $status ) || is_null( $internal_id ) || is_null( $payment_id ) ) {
			// Return a JSON response with http code 422
			return new WP_REST_Response( array(
				'status'	=> 'ERROR',
				'error'		=> 'Invalid request',
			), 422 );
		}

		// If the status is not 'INITIATED', return a JSON response with http code 200
		if ( $status !== 'INITIATED' ) {
			return new WP_REST_Response( array(
				'status' => 'OK',
			), 200 );
		}

		// Get the order ID based on the order key
		$order_id = wc_get_order_id_by_order_key( $internal_id );

		// If the order ID is 0, return a JSON response with http code 422
		if ( ! $order_id ) {
			return new WP_REST_Response( array(
				'status'	=> 'ERROR',
				'error'		=> 'Invalid request',
			), 422 );
		}

		// Get the order
		$order = wc_get_order( $order_id );

		// Add or update the _transaction_id meta
		$order->update_meta_data( '_transaction_id', $payment_id );

		// Save the order
		$order->save();

		// Mark the payment as complete
		$order->payment_complete();

		// Trigger the digiteal payment complete action
		do_action( 'clever_finance_payment_complete', $order );

		// Return a JSON response with http code 200
		return new WP_REST_Response( array(
			'status' => 'OK',
		), 200 );
	}

	/**
	 * This method sends a call to Google Analytics to track the payment
	 * @param mixed $order
	 * @return void
	 */
	public function payment_complete( $order ) {
		// Get the transaction ID
		$transaction_id = $order->get_transaction_id();

		// Get the order total
		$order_total = $order->get_total();

		// Get the order currency
		$order_currency = $order->get_currency();

		// Get the order key
		$order_key = $order->get_order_key();

		// Get the order ID
		$order_id = $order->get_id();

		// Send the call to Google Analytics
		$this->send_ga4_event( $transaction_id, $order_total, $order_currency, $order_key, $order_id );
	}

	private function send_ga4_event( $transaction_id, $order_total, $order_currency, $order_key, $order_id ) {
		

		// Get the store URL
		$store_url = get_site_url();

		// Define the request data
		$body = array(
			'client_id' => $transaction_id,
			'events' => array(
				array(
					'name' => 'purchase',
					'params' => array(
						'value' => $order_total,
						'currency' => $order_currency,
						'transaction_id' => $transaction_id,
						'items' => array(
							array(
								'item_id' => $order_key,
								'item_name' => 'Order #' . $order_id,
								'quantity' => 1,
								'price' => $order_total,
								'affiliation' => $store_url,
							),
						),
					),
				),
			),
		);
	}

}
